#!/usr/bin/env node

const express = require('express');
const { exec } = require('child_process');
const fs = require('fs');
const path = require('path');
const cors = require('cors');

const app = express();
const PORT = process.env.PORT || 3001;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.static('.'));

// Serve the main HTML file at root
app.get('/', (req, res) => {
    res.sendFile(path.join(__dirname, 'index.html'));
});

// API endpoint for knowledge risk analysis
app.post('/api/knowledge-risk', async (req, res) => {
    const { repoPath, modulePath } = req.body;
    
    if (!repoPath || !modulePath) {
        return res.status(400).send('Repository path and module path are required');
    }
    
    // Map local paths to container paths
    let containerPath = repoPath;
    if (repoPath.startsWith('/Users/')) {
        containerPath = repoPath.replace('/Users/', '/mnt/repositories/');
    }
    
    try {
        // Git command to analyze module ownership
        const gitCommand = `cd "${containerPath}" && git log --format="%an|%ad|%s" --date=short --numstat -- "${modulePath}" | head -1000`;
        
        exec(gitCommand, { maxBuffer: 1024 * 1024 * 2 }, (error, stdout, stderr) => {
            if (error) {
                return res.status(500).send(`Git command failed: ${error.message}`);
            }
            
            const analysis = analyzeModuleOwnership(stdout, modulePath);
            res.json(analysis);
        });
        
    } catch (error) {
        res.status(500).send(`Error: ${error.message}`);
    }
});

// API endpoint to generate git data for a repository
app.post('/api/generate-git-data', async (req, res) => {
    const { repoId, repoPath } = req.body;
    
    if (!repoId || !repoPath) {
        return res.status(400).send('Repository ID and path are required');
    }
    
    console.log('Original path:', repoPath);
    
    // Map local paths to container paths
    let containerPath = repoPath;
    if (repoPath.startsWith('/Users/')) {
        containerPath = repoPath.replace('/Users/', '/mnt/repositories/');
    }
    
    console.log('Container path:', containerPath);
    
    // Validate that the path exists and is a git repository
    if (!fs.existsSync(containerPath)) {
        console.log('Path does not exist:', containerPath);
        return res.status(400).send(`Repository path does not exist: ${repoPath} (mapped to ${containerPath})`);
    }
    
    const gitDir = path.join(containerPath, '.git');
    if (!fs.existsSync(gitDir)) {
        return res.status(400).send('Path is not a git repository');
    }
    
    try {
        console.log(`Generating git data for repository: ${repoPath} -> ${containerPath}`);
        
        // Generate git log data using container path
        const gitCommand = `cd "${containerPath}" && git log --pretty=format:"%H|%an|%ad|%s%n%b" --date=iso --numstat --all`;
        
        exec(gitCommand, { maxBuffer: 1024 * 1024 * 10 }, (error, stdout, stderr) => {
            if (error) {
                console.error('Git command error:', error);
                return res.status(500).send(`Git command failed: ${error.message}`);
            }
            
            if (stderr) {
                console.warn('Git command stderr:', stderr);
            }
            
            // Save the data to a repository-specific file
            const outputFile = path.join(__dirname, `git_data_${repoId}.txt`);
            
            fs.writeFile(outputFile, stdout, 'utf8', (writeError) => {
                if (writeError) {
                    console.error('File write error:', writeError);
                    return res.status(500).send(`Failed to write data file: ${writeError.message}`);
                }
                
                console.log(`Git data saved to: ${outputFile}`);
                res.json({ 
                    success: true, 
                    message: 'Git data generated successfully',
                    dataFile: `git_data_${repoId}.txt`,
                    commitCount: (stdout.match(/\|/g) || []).length
                });
            });
        });
        
    } catch (error) {
        console.error('Error generating git data:', error);
        res.status(500).send(`Error: ${error.message}`);
    }
});

// API endpoint to list available data files
app.get('/api/data-files', (req, res) => {
    try {
        const files = fs.readdirSync(__dirname)
            .filter(file => file.startsWith('git_data_') && file.endsWith('.txt'))
            .map(file => ({
                filename: file,
                repoId: file.replace('git_data_', '').replace('.txt', ''),
                size: fs.statSync(path.join(__dirname, file)).size,
                modified: fs.statSync(path.join(__dirname, file)).mtime
            }));
        
        res.json(files);
    } catch (error) {
        console.error('Error listing data files:', error);
        res.status(500).send(`Error: ${error.message}`);
    }
});

// API endpoint to delete a data file
app.delete('/api/data-files/:repoId', (req, res) => {
    const { repoId } = req.params;
    const filename = `git_data_${repoId}.txt`;
    const filepath = path.join(__dirname, filename);
    
    try {
        if (fs.existsSync(filepath)) {
            fs.unlinkSync(filepath);
            res.json({ success: true, message: 'Data file deleted successfully' });
        } else {
            res.status(404).send('Data file not found');
        }
    } catch (error) {
        console.error('Error deleting data file:', error);
        res.status(500).send(`Error: ${error.message}`);
    }
});

// API endpoint to get current config
app.get('/api/config', (req, res) => {
    try {
        const configPath = path.join(__dirname, 'config.js');
        const configContent = fs.readFileSync(configPath, 'utf8');
        res.json({ success: true, config: configContent });
    } catch (error) {
        console.error('Error reading config:', error);
        res.status(500).send(`Error reading config: ${error.message}`);
    }
});

// API endpoint to update config
app.post('/api/config', (req, res) => {
    try {
        const { config } = req.body;
        if (!config) {
            return res.status(400).send('Config content is required');
        }
        
        const configPath = path.join(__dirname, 'config.js');
        fs.writeFileSync(configPath, config, 'utf8');
        
        console.log('Config updated successfully');
        res.json({ success: true, message: 'Configuration updated successfully' });
    } catch (error) {
        console.error('Error updating config:', error);
        res.status(500).send(`Error updating config: ${error.message}`);
    }
});

// API endpoint to get repositories
app.get('/api/repositories', (req, res) => {
    try {
        const repoFile = path.join(__dirname, 'repositories.json');
        if (fs.existsSync(repoFile)) {
            const repos = JSON.parse(fs.readFileSync(repoFile, 'utf8'));
            res.json({ success: true, repositories: repos });
        } else {
            res.json({ success: true, repositories: [] });
        }
    } catch (error) {
        console.error('Error reading repositories:', error);
        res.status(500).send(`Error reading repositories: ${error.message}`);
    }
});

// API endpoint to scan folder for git repositories
app.post('/api/scan-repositories', (req, res) => {
    const { folderPath } = req.body;
    
    if (!folderPath) {
        return res.status(400).send('Folder path is required');
    }
    
    let containerPath = folderPath;
    if (folderPath.startsWith('/Users/')) {
        containerPath = folderPath.replace('/Users/', '/mnt/repositories/');
    }
    
    try {
        if (!fs.existsSync(containerPath)) {
            return res.status(400).send(`Folder path does not exist: ${folderPath}`);
        }
        
        const repositories = [];
        const items = fs.readdirSync(containerPath);
        
        for (const item of items) {
            const itemPath = path.join(containerPath, item);
            const originalPath = path.join(folderPath, item);
            
            if (fs.statSync(itemPath).isDirectory()) {
                const gitDir = path.join(itemPath, '.git');
                if (fs.existsSync(gitDir)) {
                    repositories.push({
                        id: item.toLowerCase().replace(/[^a-z0-9]/g, '_'),
                        name: item,
                        path: originalPath
                    });
                }
            }
        }
        
        res.json({ success: true, repositories });
    } catch (error) {
        console.error('Error scanning repositories:', error);
        res.status(500).send(`Error scanning repositories: ${error.message}`);
    }
});

// API endpoint to save repositories
app.post('/api/repositories', (req, res) => {
    try {
        const { repositories } = req.body;
        if (!repositories) {
            return res.status(400).send('Repositories data is required');
        }
        
        const repoFile = path.join(__dirname, 'repositories.json');
        fs.writeFileSync(repoFile, JSON.stringify(repositories, null, 2), 'utf8');
        
        console.log('Repositories saved successfully');
        res.json({ success: true, message: 'Repositories saved successfully' });
    } catch (error) {
        console.error('Error saving repositories:', error);
        res.status(500).send(`Error saving repositories: ${error.message}`);
    }
});

// Health check endpoint
app.get('/api/health', (req, res) => {
    res.json({ 
        status: 'ok', 
        timestamp: new Date().toISOString(),
        version: '1.0.0'
    });
});

// Start server
app.listen(PORT, () => {
    console.log(`🚀 Git Analytics Dashboard Server running on http://localhost:${PORT}`);
    console.log(`📊 Dashboard available at: http://localhost:${PORT}`);
    console.log(`🔧 API endpoints:`);
    console.log(`   POST /api/generate-git-data - Generate git data for a repository`);
    console.log(`   POST /api/scan-repositories - Scan folder for git repositories`);
    console.log(`   GET  /api/data-files - List available data files`);
    console.log(`   DELETE /api/data-files/:repoId - Delete a data file`);
    console.log(`   GET  /api/health - Health check`);
});

// Helper function to analyze module ownership
function analyzeModuleOwnership(gitOutput, modulePath) {
    const lines = gitOutput.trim().split('\n');
    const ownership = {};
    const commits = [];
    let currentCommit = null;
    
    for (const line of lines) {
        if (line.includes('|') && !line.match(/^\d+\s+\d+/)) {
            // Commit line: author|date|message
            const parts = line.split('|');
            if (parts.length >= 3) {
                currentCommit = {
                    author: parts[0].trim(),
                    date: parts[1].trim(),
                    message: parts.slice(2).join('|').trim(),
                    linesChanged: 0
                };
            }
        } else if (line.match(/^\d+\s+\d+/) && currentCommit) {
            // Numstat line: additions deletions filename
            const [additions, deletions] = line.split('\t');
            const linesChanged = (parseInt(additions) || 0) + (parseInt(deletions) || 0);
            currentCommit.linesChanged += linesChanged;
            
            if (!ownership[currentCommit.author]) {
                ownership[currentCommit.author] = {
                    commits: 0,
                    linesChanged: 0,
                    firstCommit: currentCommit.date,
                    lastCommit: currentCommit.date
                };
            }
            
            ownership[currentCommit.author].commits++;
            ownership[currentCommit.author].linesChanged += linesChanged;
            ownership[currentCommit.author].lastCommit = currentCommit.date;
            
            commits.push({ ...currentCommit });
            currentCommit = null;
        }
    }
    
    // Calculate risk metrics
    const totalCommits = Object.values(ownership).reduce((sum, dev) => sum + dev.commits, 0);
    const totalLines = Object.values(ownership).reduce((sum, dev) => sum + dev.linesChanged, 0);
    
    const ownershipData = Object.entries(ownership)
        .map(([author, data]) => ({
            author,
            commits: data.commits,
            linesChanged: data.linesChanged,
            commitPercentage: Math.round((data.commits / totalCommits) * 100),
            linesPercentage: Math.round((data.linesChanged / totalLines) * 100),
            firstCommit: data.firstCommit,
            lastCommit: data.lastCommit
        }))
        .sort((a, b) => b.linesChanged - a.linesChanged);
    
    // Calculate bus factor (number of people who own 50% of the code)
    let cumulativePercentage = 0;
    let busFactor = 0;
    for (const dev of ownershipData) {
        cumulativePercentage += dev.linesPercentage;
        busFactor++;
        if (cumulativePercentage >= 50) break;
    }
    
    // Risk assessment
    const topDeveloper = ownershipData[0];
    const riskLevel = topDeveloper?.linesPercentage > 70 ? 'HIGH' : 
                     topDeveloper?.linesPercentage > 50 ? 'MEDIUM' : 'LOW';
    
    return {
        modulePath,
        totalCommits,
        totalLines,
        busFactor,
        riskLevel,
        topContributor: topDeveloper,
        ownership: ownershipData,
        recentActivity: commits.slice(0, 10)
    };
}

// Graceful shutdown
process.on('SIGINT', () => {
    console.log('\n🛑 Shutting down Git Analytics Dashboard Server...');
    process.exit(0);
});

process.on('SIGTERM', () => {
    console.log('\n🛑 Shutting down Git Analytics Dashboard Server...');
    process.exit(0);
});